#' Robust Cholesky decomposition
#' 
#' Robust to singularity of its argument.
#' 
#' Like chol, but 60 times slower.  The only advantage is that it
#' produces u.t. `w` with `crossprod(w) = x`works even when `x` is p.s.d. but not p.d.
#'
#' @param x A square, positive semi-definite matrix.
#'
#' @return An upper triangular matrix `w` with `crossprod(w) == x`.  It has an
#' attribute `nerr` that is an empty vector if the matrix is positive definite,
#' a vector of the locations of zeros on the diagonal if it is only p.s.d.  If
#' `nerr` contains negative entries, `x` was not p.s.d.
#' @export
#' @md
#' 
cschol <- function(x) {
  eps <- .Machine$double.eps
  n0 <- dim(x)[1]
  m <- dim(x)[2]
  if (!n0 == m) stop("non-square argument")
  nerr <- vector("numeric", 0)
  for (i in 1:n0) {
    if (i > 1) x[i,1:(i-1)] <- 0
    w <- x[i:n0,i:n0, drop=FALSE]
    er <- 0
    n <- n0-i+1
    if (w[1,1] > 1000 * eps) {
      w[1,1] <- Re(sqrt(w[1,1]))
      if (n >= 2) {
        w[1,2:n] <- w[1,2:n]/w[1,1]
        wr <- w[1,2:n, drop=FALSE]
        w[2:n,2:n] <- w[2:n,2:n] - crossprod(wr)
      }
    } else {
      er <- 1
      if (w[1,1]<0) {
        if (w[1,1] < -1000*eps) {
          er <- -1
        }
        w[1,1] <- 0
      }
      if (n >= 2) w[2:n,1] <- 0
    }
    x[i:n0,i:n0] <- w
    if (!(er == 0)) {
      nerr <- c(nerr, i*er)
    }
  }
  w <- x
  attr(w,"nerr") <- nerr
  return(w)
}
