#' Inverse Time
#'
#' Converts `ts` object dates to integer indexes into the `ts` object
#'
#' @param tsdate Date from `ts` object
#' @param tsobj Object on whose dating the calculation is based.
#'
#' @details `tsdate` can either be a vector of real numbers, e.g. 1973.25
#'           or `c(1973.25, 1990+2/3)`, or a matrix with 2 columns containing
#'           dates in year-month pairs `c(1973,2), c(1990,8)` (for monthly data, e.g.).
#'           Safest to use 1990+2/3 for `c(1990,8)`, rather than 1990.67, as the code
#'           checks for equality of floating point numbers.
#' @export
#' @md
#' 
invTime <- function(tsdate, tsobj) {
  start <- tsp(tsobj)[1]
  end <- tsp(tsobj)[2]
  freq <- tsp(tsobj)[3]
  if (is.null(dim(tsdate)) && length(tsdate)==2) { # ambiguity
    d2 <- tsdate[1] + (tsdate[2] - 1) / freq
    d2OK <- d2 >= start   && d2 <= end 
    d1OK <- all((tsdate >= start ) & (tsdate <= end) )
    if (d2OK && d1OK) {
      warning("ambiguous date")
      return(match(d2, time(tsobj)))
    } else {
      ifelse( d2OK, match(d2, time(tsobj)), match(tsdate, time(tsobj)))
    }
  } else {
    if (is.null(dim(tsdate))) {
      return(match(tsdate, time(tsobj)))
    } else {
      tsd <- tsdate[ ,1] + (tsdate[ ,2] -1)/freq
      return(match(tsd, time(tsobj)))
    }
  }
}
