#' 2d index model likelihood
#'
#' Iterate the Kalman filter through a sample for a simple 2d index model
#'
#' @param param vector containing three rho's, two alpha's, and two observation std deviations
#' @param ydata T by 2 matrix of data (log rgdp and unemp rate / 100)
#' @param verbose when true, print out detailed results
#'
#' @return With \code{verbose=FALSE}, just minus the log likelihood.  Otherwise,
#'    \item{shat}{matrix of filtered means of the state}
#'    \item{sig}{array of filtered covariance matrices of the state}
#'    \item{lh}{log likelihood}
#'
#' @export
#' 
yuindex <-
function(param, ydata, verbose=FALSE) {
    ## param: rho (y1, y2, z), alph (y1,y2), sige (y's)
    rho <- param[1:3]
    alph <- param[4:5]
    stdy <- param[6:7]
    cnst <- param[8:9]
    G <- diag(c(rho[3], 1))
    M <- diag(c(1,0))
    N <- diag(stdy)
    H <- matrix(c(alph, cnst), nrow=2)
    ny <- dim(ydata)[1]
    sh <- matrix(0,ny, 2)
    sh[1, ] <- c(0,1)
    sg <- array(0, c(ny, 2, 2))
    sg[1, , ] <- matrix(c(400, 0, 0, 0), 2,2)
    fce <- matrix(0, ny, 2)
    lhm <- matrix(0, ny, 2)
    for ( it in 2:ny ) { step <- kf2(y=ydata[it,] - ydata[it-1,] * rho[1:2],
                                     H=H, shat=sh[it-1, ], sig=sg[it-1, , ], G=G,
                                     M=M, N=N) 
        sh[it, ] <- step$shat[1:2]
        sg[it, , ] <- step$sig[1:2,1:2]
        fce[it, ] <- step$fcsterr
        lhm[it, ] <- step$lh
    }
    if (verbose) return(list(shat=sh, sig=sg, fcsterr=fce, lh=lhm))
       else
    return(-sum(lhm))
}
