#' Kalman Filter
#'
#' Does one step of the Kalman filter recursion
#'
#' @details The model is \eqn{s = G s + t(M) e} (plant equation) and
#'          \eqn{y = H s + t(N) u} (observation equation). \eqn{e} and \eqn{u} are
#'          both \eqn{N(0,I)}.  If \eqn{N} is non-null, the returned values extend the
#'          state to include \eqn{t(N) u}, so if this function is being applied recursively, the
#'          returned shat and sig must be truncated.  It would be more efficient to
#'          rewrite the model with the expanded state vector (and hence null \eqn{N}).
#' @param y The observation vector.
#' @param H The matrix connecting observation to state.
#' @param shat The prior mean for the state.
#' @param sig The prior covariance matrix for the state.
#' @param G The matrix coefficient on the past state in the plant equation.
#' @param M The Cholesky factor of the covariance matrix of shocks to the plant equation.
#' @param N The Cholesky factor of the covariance matrix of errors in the observation equation.
#' @return A list with these elements:
#'    \describe{
#'        \item{\code{shat}}{new mean for the state}
#'        \item{\code{sig}}{new covariance matrix for the state}
#'        \item{\code{lh}}{length-2 vector of log likelihood elements (summing to log likelihood)}
#'        \item{\code{fcsterr}}{error in forecast of the observation}
#'   }
#' @export
#' 
kf2 <- function(y,H,shat,sig,G,M, N=NULL) {
  if (is.null(dim(H))) dim(H) <- c(1,length(H))
  nobs <- dim(H)[1]
  nstate <- dim(H)[2]
  if (!is.null(N)) {
     shat <- c(shat, rep(0, nobs))
     H <- cbind(H, diag(nrow=nobs))
     sigx <- matrix(0, nrow = nstate + nobs, ncol=nstate+nobs)
     sigx[ 1:nstate, 1:nstate] <- sig
     sigx[nstate + (1:nobs), nstate + (1:nobs)] <- crossprod(N)
     sig <- sigx
     Gx <- matrix(0, nrow=nstate + nobs, ncol=nstate + nobs)
     Gx[1:nstate, 1:nstate] <- G
     G <- Gx
     Mx <- matrix(0, nstate + nobs, nstate + nobs)
     Mx[1:nstate, 1:nstate] <- M
     Mx[ nstate + 1:nobs, nstate + 1:nobs] <- N
     M <- Mx
  }
  SMALLSV <- 1e-7
  omega <- G %*% sig %*% t(G) + crossprod(M)
  ## stopifnot (nstate >= nobs)
  ##------------ Don't need separate treatment of H == 0.  H %*% G %*% t(H) = 0 covers it.
  ##   if (isTRUE(all.equal(H, 0))) { # No observation case.  Just propagate the state.
  ##     lh <- c(0,0)
  ##     shatnew <- G %*% shat
  ##     signew <- omega
  ##     fcsterr <- y                        # y had better be 0
  ##     if (!all.equal(y,0) ) warning("zero H but non-zero y")
  ##   } else {
  ho <- H %*% omega
  svdhoh <- svd( ho %*% t(H) )
  if (all(svdhoh$d < SMALLSV)) { # Observation is uninformative. Propagate state.
    lh <- c(0,0)
    shatnew <- G %*% shat
    signew <- omega
    fcsterr <- y - H %*% G %*% shat     # had better be 0
    if (!all(abs(fcsterr) < 1e-7)) warning("Uninformative H but non-zero fcsterr")
  } else {
    first0 <- match(TRUE, svdhoh$d < SMALLSV)
    if (is.na(first0)) first0 <- min(dim(H))+1
    u <- svdhoh$u[ , 1:(first0-1), drop=FALSE]
    v <- svdhoh$v[ , 1:(first0-1), drop=FALSE]
    d <- svdhoh$d[1:(first0-1), drop=FALSE]
    fcsterr <- y-H %*% G %*% shat
    hohifac <- (1/sqrt(d)) * t(u)       #diag(1/sqrt(d)) %*% t(u)
    ferr <- hohifac %*% fcsterr
    lh <- c(0,0)
    lh[1] <- -.5 * crossprod(ferr)
    lh[2] <- -.5 * sum( log(d) ) - .5 * length(d) * log(2 * pi)
    ## log(2 * pi) term added 2013.12.11. corrected 2015.10.13
    ## changed dim(H)[1] to length(d) 2016.10.24
    hohoifac <-hohifac %*% ho 
    shatnew <- crossprod(hohoifac, ferr) + G %*% shat
    signew <- omega - crossprod(hohoifac)
  }
  return(list(shat=shatnew, sig=signew, lh=lh, fcsterr=fcsterr))
}
