#' qzswitch
#'
#' reorder adjacent eigenvalues in a qz decomposition
#' 
#' Takes U.T. matrices \code{a, b}, orthonormal matrices \code{q,z},
#' interchanges diagonal elements \code{i} and \code{i+1} of both
#' \code{a} and \code{b}, while maintaining \code{q \%*\% a \%*\% t(z)}
#' and \code{q \%*\% b \%*\% t(z)}
#' unchanged.  If diagonal elements of \code{a} and \code{b} 
#' are zero at matching positions, the returned \code{a} will have zeros at both 
#' positions on the diagonal.  This is natural behavior if this routine is used 
#' to drive all zeros on the diagonal of \code{a} to the lower right, but in
#' this case  the qz transformation is not unique and it is not possible simply
#' to switch the positions of the diagonal elements of both \code{a}
#' and \code{b}.
#' 
#' @param i the position on the diagonal to be switched with the one below
#' @param qzlist the list of matrices from \code{qz}, translated to the local
#'   list names as at the top of \code{gensys}.
#' @return a list like the input, but with i and i+1 positions switched.
#' @md
#' @export
#' @seealso [qzdiv()], [gensys()], QZ package.
qzswitch <- function(i=1,qzlist)
  {
    realsmall <- 1e-7; 
    ##realsmall<-1e-3;
    a <- qzlist$a
    b <- qzlist$b
    q <- qzlist$q
    z <- qzlist$z
    gev <- qzlist$gev
    q <- t(Conj(q))                     #This is needed because the code was
                                        #originally for matlab, where it is q'az' that
                                        # is preserved.
    A <- a[i,i]; d <- b[i,i]; B <- a[i,(i+1)]; e <- b[i,(i+1)]
    g <- a[i+1,i+1]; f <- b[i+1,i+1]
    w2 <- rep(0,2)
    ## a[i:(i+1),i:(i+1)]<-[A B; 0 g]
    ## b[i:(i+1),i:(i+1)]<-[d e; 0 f]
    if (abs(g)<realsmall & abs(f)<realsmall)
      {
        if (abs(A)<realsmall)
          {
            ## l.r. coincident 0's with u.l. of a<-0; do nothing
            return(list(a=a,b=b,q=q,z=z))
          } else
        {
          ## l.r. coincident zeros; put 0 in u.l. of a
          wz <- c(B, -A)
          wz <- wz/sqrt(sum(Conj(wz)*wz))
          wz <- array(c(wz,Conj(wz[2]),-Conj(wz[1])),dim=c(2,2))
          xy <- diag(2)
        }
      } else
    {
      if (abs(A)<realsmall && abs(d)<realsmall)
        {
          if (abs(g)<realsmall)
            {
              ## u.l. coincident zeros with l.r. of a<-0; do nothing
              return(list(a=a,b=b,q=q,z=z))
            } else
          {
            ## u.l. coincident zeros; put 0 in l.r. of a
            wz <- diag(2)
            xy <- c(g,-B)
            xy <- xy/sqrt(sum(xy*Conj(xy)))
            xy <- t(matrix(c(Conj(xy[2]), -Conj(xy[1]),xy),nrow=2,ncol=2))
          }
        } else
      {
        ## usual case
        wz <- c(g*e-f*B, Conj(g*d-f*A))
        xy <- c(Conj(B*d-e*A), Conj(g*d-f*A))
        n <- drop(sqrt(wz %*% Conj(wz)))
        m <- drop(sqrt(xy %*% Conj(xy)))
        ## drop function here makes n and m scalars, rather than 1x1 matrices.
        ## R started complaining about using 1x1 matrices as scalars with version 3.4.
        ## Benjamin Johannsen noticed this and proposed the fix, 2017.9.23.
        if (Re(m)<1e-12*100)
          {
            ## all elements of a and b proportional
            return(list(a=a,b=b,q=q,z=z))
          }
        wz <- wz/n
        xy <- xy/m
        wz <- matrix(c(wz, -Conj(wz[2]),Conj(wz[1])),byrow=TRUE,ncol=2,nrow=2)
        xy <- -matrix(c(xy,-Conj(xy[2]), Conj(xy[1])),byrow=TRUE,ncol=2,nrow=2)
        ## This sign flip (8/20/10) makes xy = t(Conj(wz)) when b == diag(2)
        ## Then qzswitch will work on a Schur decomposition generated by qz(a, diag(n))
      }
    }
    a[i:(i+1),] <- xy %*% a[i:(i+1),]
    b[i:(i+1),] <- xy %*% b[i:(i+1),]
    a[,i:(i+1)] <- a[,i:(i+1)] %*% wz
    b[,i:(i+1)] <- b[,i:(i+1)] %*% wz
    z[,i:(i+1)] <- z[,i:(i+1)] %*% wz
    q[i:(i+1),] <- xy %*% q[i:(i+1),]
    q <- t(Conj(q))
    w2 <- gev[i,]
    gev[i,] <- gev[i+1,]
    gev[i+1,] <- w2
    qzlist$a <- a
    qzlist$b <- b
    qzlist$q <- q
    qzlist$z <- z
    qzlist$gev <- gev
    return(qzlist)
  }
