#' Continuous time impulse responses
#'
#' Generates irf's from the output of `gensysct()`
#'
#' `impact` and `G1` are assumed to be of the form in the output of gensysct.R.  The total
#' number of points at which irf's are computed are `span / interval`.
#' 
#' @param impact Weights on the white noise exogenous shocks. Number of variables by
#'               number of shocks matrix.
#' @param G1 Matrix of coefficients on the levels of the variables in the \eqn{y}
#'           vector in the `ydot = G1 %*% y + impact %*% eps` form.
#' @param interval The time interval between the discrete values of the time index at
#'                 which the irf's are computed.
#' @param span The span of time over which the irf's are computed.  
#'
#' @return A 3-d array, with dimensions the number of variables, the number of shocks,
#' and the number of points at which the irf's have been computed.
#' 
#' @md
#' @export
#' 
impulsct <- function(impact, G1, interval, span) {
  nv <- dim(G1)[1]
  ns <- dim(impact)[2]
  nrsp <- ceiling(span/interval)+1
  resp <- array(0,c(nv, ns, nrsp))
  A <- padm(G1 * interval)
  resp[ , , 1] <- impact
  for (it in 2:nrsp) resp[ , , it] <- A %*% resp[ , , it-1]
  dimnames(resp) <- list(dimnames(impact)[[1]], dimnames(impact)[[2]], NULL)
  return(resp)
}
  
