#' Evaluate system matrices, continuous time model
#' 
#' Form numerical system matrices for continuous time linearized system
#' 
#' @param dexpr     derivative expression vector, as produced by g0g1dc
#' @param x         nx by 1 matrix of values of x, names in dimnames(x)[[1]]
#' @param xd     nx by 1 matrix of values of xdot, names in dimnames(x)[[1]]
#' @param shock:     exogenous shocks (all assumed centered at 0)
#' @param experr     logical vector, TRUE at forward-looking equations' positions.
#' @param param     vector of values of parameters (all other symbols) in the expressions
#'
#' @details NB x, xd, and param must all be vectors with named components, or lists.
#'
#' @return
#' \describe{
#'   \item{`g0`}{Coefficients on derivatives}
#'   \item{`g1`}{Coefficients on levels, on rhs}
#'   \item{`Psi`}{Coefficients on shocks}
#'   \item{`Pi`}{Coefficients on endogenous expectational errors}
#' }
#' @md
#' @export
#' 
g0g1evalc <- function(dexpr, x, xd=vector("numeric",length(x)),
                      shock=attr(dexpr,"shock"), experr=attr(dexpr,"forward"),
                      param=attr(dexpr,"param")) {
    if (is.null(names(x))) { names(x) <- dimnames(x)[[1]]}
    if (is.null(names(xd))) names(xd) <- paste(names(x),"dot",sep="")
    if(identical(shock, "NONE")) {
        shock <- vector("character", 0)
    } else {
        if(is.character(shock)) {
            nshock <- shock
            shock <- rep(0, length(nshock))
            names(shock) <- nshock
        } else {
            if (is.null(names(shock))) names(shock) <- dimnames(shock)[[1]]
        }
    }
    if (is.null(names(param))) names(param) <- dimnames(param)[[1]]
    nex <- length(dexpr)
    nx <- length(x)
    nxd <- length(xd)
    nshock <- length(shock)
    g0 <- matrix(0, nex, nxd )
    g1 <- matrix(0, nex, nx )
    Psi <- matrix(0, nex, nshock)
    nfwd <- sum(experr)
    Pi <- matrix(0, nex, nfwd)
    dimnames(Pi) <- list(names(x), NULL)
    for ( i in 1:nex ) {
        dvec <- attr(eval(eval(dexpr[i]), as.list(c(xd,x,shock,param))),"gradient")
        loc <- 0
        g0[i,] <- dvec[1:nx]
        loc <- nx
        g1[i,] <- dvec[(loc+1):(loc+nxd)]
        loc <- loc + nxd
        Psi[i,] <- dvec[(loc+1):(loc+nshock)]
    }
    Pi[experr,] <- diag(nfwd)
    dimnames(g0) <- list(names(dexpr),names(x))
    dimnames(g1) <- dimnames(g0)
    if(is.numeric(experr)) experr <- names(dexpr)[experr]
    dimnames(Psi) <- list(names(dexpr),names(shock))
    dimnames(Pi) <- list(names(dexpr), names(dexpr)[experr])
    return(list(g0=g0, g1=-g1, Psi=-Psi, Pi=-Pi, param=param))
}
