#' Panel var prior
#'
#' Create dummy observations for prior on reduced form in panel var model
#'
#' @details Each country gets dummy observations.  The ones that apply only to
#'    coefficients that do not vary across countries (all except coint dummies)
#'    are weighted by the inverse of the square root of the number of countries.
#'    This prevents the prior from tightening automatically when the number of
#'    countries increases.  The X dummies have country-specific constants first,
#'    then the lagged variables that have constant coefficients across countries.
#'    Note that the X matrix has constants in the first columns.
#'
#' @param initcond list of lags by nv matrices of initial values, one for each
#'    country. 
#' @param sig vector of initial guesses for standard deviations of disturbances,
#'            one for each variable in the system, or else a matrix, one column
#'            per country. 
#' @param tight The tightness of the prior on individual coefficients--- bigger
#'              is tighter
#' @param decay Rate of harmonic decay in prior variances with lags.  0 is no
#'              decay
#' @param urtight tightness of dummy observations that pull toward independent
#'                martingales
#' @param cointight tightness of dummy observations that pull toward
#'                  cointegration and pull constant terms toward matching
#'                  initial condition sample means. This generates the prior on
#'                  the group-specific constants.
#' @param notrw  logical vector, TRUE for variables that are not persistent, so
#'               prior center is i.i.d., not random walk.  All FALSE by default.
#' @param condOn Which variables are treated as exogenous (entering X, but not Y).
#'
#' @return A list, indexed by country, of dummy variable `Y` and `X` matrices (a
#'         two-elment list for each country).  The list has a `priorpars`
#'         aattribute that lists all the prior parameters.
#' 
pvarprior <- function(initcond, lags, sig, tight, decay, urtight, cointight, notrw=NULL, condOn=NULL) {
    if (is.null(dim(initcond[[1]]))) {
        initcond <- lapply(initcond, function(x) matrix(x, ncol=1))
    }
    nv <- dim(initcond[[1]])[2]
    nc <- length(initcond)
    rnc <- sqrt(nc)
    ## lags <- dim(initcond[[1]])[1]
    ## lags now an argument, so initcond argument can be the same data used for estimation
    for (ic in 1:nc) initcond[[ic]] <- initcond[[ic]][1:lags, , drop=FALSE]      
    country <- names(initcond)
    if (is.null(notrw)) notrw <- rep(FALSE, nv)
    Endog <- setdiff(1:nv, condOn)
    ncondn <- length(condOn)
    nendog <- nv - ncondn
    ndobs <- 1 + nv + lags * nv + ncondn * nendog     #per country
    #notrw <- notrw[Endog]
    Y <- matrix(0, ndobs, nendog)
    X <- matrix(0, ndobs, lags * nv + nc + ncondn)
    YXout <- list()
    for ( ic in 1:nc ) {
        thisic <- initcond[[ic]]
        Y[ , ] <- 0
        X[ , ] <- 0
        ## coint prior
        ##------------
        ## All variables, including condnOn, at their initial means
        ## implies little change in Y
        ## Because this generates the priors on constants, it is not weighted by 1/rnc.
        meanic <- apply(thisic, 2, mean)
        Y[1 , ] <-  meanic[Endog] * cointight 
        X[1 , nc + (1:(nv * lags))] <- rep(meanic , lags) * cointight
        X[1 , ic] <- cointight
        ##-------------
        ## unit root prior
        if (nv > 1) {
            Y[2:(nv + 1), ] <- diag(meanic, nv,nv)[,Endog] * urtight / rnc
            Y[2:(nv + 1),  ][notrw, ] <- 0
        } else {
            Y[2, ] <- meanic[Endog] * urtight / rnc
        }
        
        X[2:(nv + 1), 1:nc] <- 0
        X[2:(nv + 1), nc + (1:(nv * lags))] <- t(apply(diag(meanic, nv,nv), 1, rep, lags)) * urtight / rnc
        if ( nv > 1 ) {
            X[2:(nv + 1) , ][notrw, ] <- 0
        } else {
            if ( notrw ) X[2, ]  <- 0
        }
        ## coeff-by-coeff MN prior
        mndx <- nv + 1 + (1: (nv * lags))
        if ( nv > 1) {
            Y[mndx, ] <- diag(sig, nv * lags, nv)[,Endog] * tight / rnc
            Y[mndx, ][notrw, ] <- 0
        } else {
            Y[mndx, ] <- c(sig, rep(0, lags - 1)) * tight / rnc
        }
        for (iv in 1:nv) {
            for (il in 1:lags) {
                X[nendog + 1 + (il - 1) * nv + iv, nc + (il - 1) * nv + iv] <- sig[iv] * il^decay * tight/rnc
            }
        }
        ## prior on contemporary coeffs of condOn
        ##------------------------------------------
        if (ncondn>0) {
            cdn0base <- nv + 1 + nv * lags
            Y[cdn0base + 1:(ncondn * nendog), ] <- t(matrix(diag(sig[Endog]), nendog, nendog * ncondn))
            for (icdn in 1:ncondn) {
                X[cdn0base + (icdn - 1) * nendog + 1:nendog, nc + nv * lags + icdn] <- sig[condOn[icdn]]
            }
        }
        ##-----------------------------------------
        YXout[[country[ic]]] <- list(Y=Y, X=X)       
    }
    attr(YXout, "priorpars") <- list(sig=sig, tight=tight, decay=decay, urtight=urtight, cointight=cointight, condOn=condOn)
    return(YXout)
}
#' @export
#' @md
