#' Panel structural VAR estimation
#'
#' Returns marginal data density conditional on `A` and `lambda`
#'
#' Using actual and dummy observations stacked in the first argument,
#' the `A` and `lambda` arguments are used to convert the system to a set of
#' weighted-data normal linear regression models.  The model is
#' \deqn{A(L) y_t = c_i \Lambda_i\epsilon_t,}
#' where \eqn{\epsilon} is \eqn{N(0,I)} and independent of past \eqn{y}.
#' The data are in groups \eqn{i} corresponding to, e.g., a country or a time
#' span.  They should be, within each group, a multiple time series object.
#' The  groups can be of different sizes.
#'
#' @param YX The stacked `Y` and `X` data returned by [StackData()].  
#' @param A The matrix of coefficients on contemporaneous variables in the SVAR.
#' @param lambda The matrix of scale factors for the structural shocks, which
#' vary across groups and across variables
#'
#' @return   The list components returned are
#' \describe{
#'    \item{lmdd}{log marginal data density.}
#'    \item{mout}{Returned list from applying `mgls()` to all the data.}
#'    \item{priormout}{Returned list from applying `mgls` to prior dummy
#'       observations alone.}
#'    \item{lags}{Number of lags.}
#'    \item{dumspan}{Number of dummy observations in each group.}
#'    \item{call}{The function call that generated this output}
#' }
#' The returned list is of S3 class `psv`. The utilities [coef.psv()],
#'    [residuals.psv()], and [vcov.psv()] extract the more commonly used
#'    results in convenient form.
#' @seealso [pvarprior()] (to set the prior conditional on `A` and `lambda`),
#'   [StackData()] (to combine the dummy observations from `pvarprior()` with the
#'   real data. 
pSVARmdd <- function(YX, A, lambda) {
    nc <- length(YX$tspReal)
    lags <- YX$lags
    realspans <- unlist(lapply(YX$tspReal, function(x) (x[2] - x[1]) * x[3] + 1 -lags))
    ##tspReal spans initial conds.
    ## tsp is (start, end, freq)
    dumspan <- YX$countryEnds[1] - realspans[1]
    isdum <- dumspan > 0
    if (isdum) {
        dumndx <- unlist(lapply(YX$countryEnds, function(x) seq(x - dumspan + 1, x)))
        dumout <- mgls(YX$Y[dumndx, ], YX$X[dumndx, ], A, lambda, breaks=dumspan * (1:nc),
                       verbose=TRUE)
    } else {
        dumout <- list(lmdd=NULL)
    }
    fullout <- mgls(YX$Y, YX$X, A, lambda, breaks=YX$countryEnds, verbose=TRUE)
    lmdd <- fullout$lmdd - if (isdum) dumout$lmdd else 0
    rv <- list(lmdd=lmdd, mout=fullout, priormout=dumout, lags=lags, dumspan=dumspan, call=match.call())
    class(rv) <- c("psv", "list")               
    return(rv)
}
#' @export
#' @md
