#' multivariate block GLS
#'
#' Multiple left-hand sides, all transformed by A to structural form, weighted in blocks by `lambda`
#'
#' @details Model is
#' \deqn{Y_i A \Lambda_i = X_i B + e_i}
#' where A is common across groups, \eqn{Y_i} and \eqn{X_i} are blocks of rows of Y and X corresponding
#' to group i, and \eqn{\Lambda_i} is a diagonal matrix.  The model assumes the e_i are N(0,I) and uncorrelated
#' across i.  If \code{breaks=NULL}, there is no group structure (i.e. just one group).  The names of
#' the columns of A propagate to name the columns of \code{uraw} and \code{B}  The \code{XR} returned
#' value is needed to form posterior variances of B.
#'
#' @param Y the dependent data matrix
#' @param X the right-hand side data matrix
#' @param A the matrix transformation that makes the errors independent across
#'    equations
#' @param lambda the nv by ng (number of variables by number of groups) matrix,
#'    each column of which scales the residuals in the corresponding group to
#'    have unit variance
#' @param breaks the vector  of integers marking the ends of the groups as one
#'    goes down the rows of X or Y
#' @param verbose If true, return key inputs as well as the outputs calculated
#'    within the routine
#'
#' @return \describe{
#' \item{llh}{log likelihood conditional on \code{A,lambda}, maximized over
#'    \code{B}}
#' \item{lmdd}{log of marginal data density, i.e. posterior integrated over B}
#' \item{B}{matrix of regression coefficients, each column corresponding to one
#'    equation}
#' \item{uraw}{residuals that theoretically are i.i.d. N(0,1)}
#' \item{XR}{R piece of QR decomposition of X.  unless \code{breaks=NULL}, a
#'    list with one term for each variable}
#' }
mgls <- function(Y, X, A=diag(dim(Y)[2]), lambda, breaks=NULL, verbose=FALSE) {
    snames <- dimnames(A)[[2]]
    Y <-  Y %*% A
    nv <- dim(Y)[2]
    nx <- dim(X)[2]
    nobs <- dim(Y)[1]
    B <- matrix(0, nx, nv)
    Xwork <- X
    XR <- list()
    uraw <- matrix(0, nobs, nv)
    if (!is.null(breaks)) {
        nc <- length(breaks)
        breaks <- c(0, breaks)
        clengths <- diff(breaks)
        lambdamat <- matrix(0, nv, nobs)
        for (ic in 1:nc) lambdamat[ , (breaks[ic] + 1):breaks[ic + 1]] <- lambda[ , ic]
        Y <- Y * t(lambdamat)
        for (iv in 1:nv) {
            Xwork <- c(lambdamat[iv, ]) * X
            qrx <- qr(Xwork)
            XR[[iv]] <- qr.R(qrx)
            attr(XR[[iv]], "pivot") <- qrx$pivot
            B[ , iv] <- qr.coef(qrx, Y[ , iv])
            uraw[ , iv] <- qr.resid(qrx, Y[ , iv])
        }
    } else {                            #breaks was null
        breaks <- c(0, nobs)
        qrx <- qr(X)
        XR <- qr.R(qrx)
        attr(XR, "pivot") <- qrx$pivot
        B <- qr.coef(qrx, Y)
        uraw <- qr.resid(qrx, Y)
        lambdamat <- 1
        XR <- list(XR)
    }
    llh <- nobs * determinant(A)$modulus + sum(log(lambdamat)) - log(2 * pi) * nobs * nv / 2 -
                               .5 * sum(uraw^2)
    lmdd <- llh - sum(log(abs(unlist(lapply(XR, diag))))) + log(2 * pi) * nx * nv /2
    if (verbose) 
        return(list(llh=llh, lmdd=lmdd, B=B, uraw=uraw, XR=XR, A=A, lambda=lambda, breaks=breaks))
    else
        return(list(llh=llh, lmdd=lmdd, B=B, uraw=uraw, XR=XR))
}
#' @export
    
        
            
    
