#' Construct Y and X matrices for VAR regression from a list of ts objects
#'
#' Model is a VAR, with some elements of \code{Y} possibly conditioned on, so treated
#' as exogenous.
#' 
#' The `ts` objects in the `realObs` list already have NAs removed from ends,
#' have been logged where necessary etc,  and contain just the desired set of 
#' variables. The dumObs list contains the already-formed dummy observations 
#' to implement the prior.
#'
#' @param realObs list of data time series (`mts`) objects, indexed by country name
#' @param dumObs list of dummy observation `Y` and `X` objects as produced by
#'               `pvarprior`.
#' @param lags number of lags
#' @param condOn list of variables being conditioned on.
#' @return \describe{
#' \item{Y,X} {left-hand-side and right-hand side matrices, including country constants
#'      and current and lagged values of variables conditioned on}
#' \item {tspReal}{Actual time span of data, including lags, for each country}
#' \item {countryEnds}{last row of Y,X for each country}
#' \item {lags}{number of lags}
#' \item {priorpars}{prior parameters passed through from dumObs attribute}
#' }
StackData <- function(realObs, dumObs=NULL, lags, condOn=NULL) {
    nc <- length(realObs)
    cnames <- names(realObs)
    vnames <- dimnames(realObs[[1]])[[2]]
    if (!is.null(dumObs)) stopifnot(nc == length(dumObs))
    nv <- dim(realObs[[1]])[2]
    ncondn <- length(condOn)
    Endog <- setdiff(1:nv, condOn)
    nrobs <- vector("numeric", nc)
    ndobs <- if(is.null(dumObs)) 0 else vector("numeric", nc)
    for (ic in 1:nc) {
        nrobs[ic] <- dim(realObs[[ic]])[1] - lags
        ndobs[ic] <- if(is.null(dumObs))  0 else dim(dumObs[[ic]]$Y)[1]
    }
    nrobsT <- sum(nrobs)
    ndobsT <- sum(ndobs)
    Y <- matrix(0, nrobsT + ndobsT, nv - ncondn)
    X <- matrix(0, nrobsT + ndobsT, nv * lags + nc + ncondn) # separate constant terms in 1st columns
    countryEnds <- nrobs + ndobs
    rowno <- 0
        for (ic in 1:nc) {
        Y[rowno + (1:nrobs[ic]), ] <- realObs[[ic]][-(1:lags), Endog]
        X[rowno + (1:nrobs[ic]), ic] <- 1 # country=specific constant
        for (il in 1:lags) {
            X[rowno + (1:nrobs[ic]), nc + (il-1) * nv  + (1:nv)] <- realObs[[ic]][(1:nrobs[ic]) + lags - il, ]
        }
        if (ncondn > 0) {
            X[rowno + (1:nrobs[ic]), nc + nv * lags + 1:ncondn] <- realObs[[ic]][(1:nrobs[ic]) + lags, condOn]
        }
        if (!is.null(dumObs) ) {
            Y[rowno + nrobs[ic] + (1:ndobs[ic]), Endog] <- dumObs[[ic]]$Y
            X[rowno + nrobs[ic] + (1:ndobs[ic]), -(1:nc)] <- dumObs[[ic]]$X[ , -(1:nc)]
            X[rowno + nrobs[ic] + (1:ndobs[ic]), ic] <- dumObs[[ic]]$X[ , ic]
        }
        rowno <- rowno + countryEnds[ic]
    }
    tspReal <- lapply(realObs, tsp)
    names(countryEnds) <- names(realObs)
    dimnames(Y) <- list(NULL, vnames[Endog])
    xnames <- vector("character", nc + nv * lags + ncondn)
    xnames[1:nc] <- paste("const", names(realObs), sep="_")
    xnames[nc + 1:(nv * lags)] <- paste(rep(unlist(vnames), lags), rep(as.character(1:lags), each=nv),sep="")
    if (ncondn>0) {
        xnames[nc + nv * lags + 1:ncondn] <- paste(unlist(vnames)[condOn], as.character(0),sep="")
    }
    dimnames(X)[[2]] <- xnames
    countryEnds <- cumsum(countryEnds)
    priorpars <- if (is.null(dumObs)) NULL else attr(dumObs, "priorpars")
    return(list(Y=Y, X=X, tspReal=tspReal, countryEnds=countryEnds, lags=lags, condOn=condOn, priorpars=priorpars))
}
#' @export
#' @md
